/*** CONFIDENTIAL ***/
/* Copyright (C) 2011 2012 2013, Panasonic Corporation */
#include "avev3_linux.h"
#include "avev3_machine.h"
#include "avev3.h"



#define AVEV3_CLKCR_OFF (0)

static void avev3_wait_phy_ready(void);
static volatile int avev3_initial_global_reset_completed = 0; 

static const char *drv_version =
  "avev3 drv: $Revision: 2.00 \n";

static int debug = -1;    

unsigned long avev3_reg_base;

#ifdef AVEV3_ENABLE_LONGWAIT_BY_SLEEP 
unsigned long avev3_flg_reset = 0;  
#endif 

#ifdef AVEV3_USE_PHY_RTL8201 
int avev3_phy_reset_period    =  20*1000;  
int avev3_phy_reset_stability = 280*1000;  
#else 
int avev3_phy_reset_period    = 1000;      
int avev3_phy_reset_stability = 1000;      
#endif 

#ifdef AVEV3_ENABLE_LONGWAIT_BY_SLEEP
#define AVEV3_WAIT_FOR_PHY_RESET(usec) ((usec) < 1000) ? udelay((usec)) : msleep((usec)/1000)
#else 
#define AVEV3_WAIT_FOR_PHY_RESET(usec) ((usec) < 1000) ? udelay((usec)) : mdelay((usec)/1000)
#endif 

#ifdef AVEV3_ENABLE_POL_LINKCHK 
extern int avev3_poll_link;   
#endif 

static int avev3_base = LAN_CSBASE;                  
static int avev3_irq  = _AVEV3_IRQ;                  


MODULE_AUTHOR("Maintainer: AVEV3 soft dev.");
MODULE_DESCRIPTION("Panasonic AV-EtherV3 driver");
MODULE_LICENSE("Original");

module_param(debug, int, 0);
MODULE_PARM_DESC(debug, "debug level (0-6)");
#if 0
#define DEBUG_DEFAULT   (NETIF_MSG_DRV | NETIF_MSG_HW | NETIF_MSG_RX_ERR | \
             NETIF_MSG_TX_ERR)
#endif


#define DEBUG_DEFAULT   (0)     


int avev3_rx(struct net_device *dev, int ring, int num);
int avev3_tx_completion(struct net_device *dev);


struct net_device *avev3_dev = NULL;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,32) 
static struct net_device_ops avev3_netdev_ops;
#endif 

struct AVEV3_proc_info avev3_info[] =
  {
    {"RxFIFO OVF   ", 0},   

    {"resrve       ", 0},
    {"resrve       ", 0},
    {"resrve       ", 0},
    {"VLAN Tag     ", 0},
    {"Frame Bad    ", 0},   
    {"Frame Good   ", 0},   

    {"Tx Bytes     ", 0},
    {"Rx Bytes     ", 0},
    {"RxD recv     ", 0},   
    {"resrve       ", 0},   
    {"resrve       ", 0},   
    {"resrve       ", 0},   
    {"resrve       ", 0},   
    {"RxD drop     ", 0},   
    {"resrve       ", 0},   
    {"resrve       ", 0},    
    {"resrve       ", 0},   
    {"resrve       ", 0},    
    {"resrve       ", 0},   
    {""             , 0}    
  };




#define RUN_AT(x) (jiffies + (x))

unsigned int avev3_startup_irq(unsigned int irq)
{
  return 0;
}
void avev3_shutdown_irq(unsigned int irq)
{
  return;
}
void avev3_ack_irq(unsigned int irq)
{
}
void avev3_end_irq(unsigned int irq)
{
}


int avev3_mdio_read(struct net_device *dev, int phy_id, int location)
{
  struct avev3_private *priv = netdev_priv(dev);
  unsigned long mdio_ctl;
  unsigned long reg = 0;
  long          loop_cnt = 0;

  if ( netif_msg_drv(priv) ) {
    DBG_PRINT("avev3_mdio_read() phyid=0x%02x addr=0x%02x\n", phy_id, location);
  }

#ifndef AVEV3_PHY_AUTODETECT
  if ( phy_id != AVEV3_PHY_ID ) {
    printk(KERN_ERR "%s: PHY ID [%d] is wrong.\n", dev->name, phy_id);
    return (int)reg;
  }
#endif
    
#if defined(AVEV3_SEPARATE_WAIT_FOR_PHY_READY)
  if ( avev3_initial_global_reset_completed == 0 ) {
    avev3_wait_phy_ready();
  }
#endif  

  AVEV3_REG_READ(mdio_ctl, _AVEV3_MDIOCTR);

  AVEV3_REG_WRITE((((phy_id & priv->mii_if.phy_id_mask) << 8) |
              (location & priv->mii_if.reg_num_mask)), _AVEV3_MDIOAR);

  wmb();       
  AVEV3_REG_WRITE((mdio_ctl | _AVEV3_MDIOCTR_RREQ), _AVEV3_MDIOCTR);

  loop_cnt = 0;
  while (1) {
    AVEV3_REG_READ(reg, _AVEV3_MDIOSR);
    if ( !(reg & _AVEV3_MDIOSR_STS) ) {
      break;
    }
    udelay(10); 
    loop_cnt++;
    if( loop_cnt > _AVEV3_MDIO_MAXLOOP) {
      printk(KERN_ERR "%s: MDIO read error (MDIOSR=0x%08lx).\n", dev->name, reg);
      return 0;
    }
  }

  AVEV3_REG_READ(reg, _AVEV3_MDIORDR);

  if ( netif_msg_drv(priv) ) {
    DBG_PRINT("  =====> data=0x%04lx\n", reg);
  }

  return (int)reg;
}

void    avev3_mdio_write (struct net_device *dev, int phy_id, int location,
             unsigned long val)
{
  struct avev3_private *priv = netdev_priv(dev);
  unsigned long mdio_ctl;
  unsigned long reg;
  unsigned long reg_rxcr, reg_rxcr_new, reg_txcr_new;
  long          loop_cnt = 0;

  if ( netif_msg_drv(priv) ) {
    DBG_PRINT("avev3_mdio_write() phyid=0x%02x addr=0x%02x data=0x%04lx\n", phy_id, location, val);
  }

#ifndef AVEV3_PHY_AUTODETECT
  if ( phy_id != AVEV3_PHY_ID ) {
    printk(KERN_ERR "%s: PHY ID [%d] is wrong.\n", dev->name, phy_id);
    return;
  }
#endif
    
#if defined(AVEV3_SEPARATE_WAIT_FOR_PHY_READY)
  if ( avev3_initial_global_reset_completed == 0 ) {
    avev3_wait_phy_ready();
  }
#endif  

  AVEV3_REG_READ(mdio_ctl, _AVEV3_MDIOCTR);

  AVEV3_REG_WRITE((((phy_id & priv->mii_if.phy_id_mask) << 8) |
              (location & priv->mii_if.reg_num_mask)), _AVEV3_MDIOAR);

  AVEV3_REG_WRITE(val, _AVEV3_MDIOWDR);

  wmb();       
  AVEV3_REG_WRITE((mdio_ctl | _AVEV3_MDIOCTR_WREQ), _AVEV3_MDIOCTR);

  loop_cnt = 0;
  while (1) {
    AVEV3_REG_READ(reg, _AVEV3_MDIOSR);
    if ( !(reg & _AVEV3_MDIOSR_STS) ) {
      break;
    }
    udelay(10); 
    loop_cnt++;
    if( loop_cnt > _AVEV3_MDIO_MAXLOOP) {
      printk(KERN_ERR "%s: MDIO write error (MDIOSR=0x%08lx).\n", dev->name, reg);
      break;
    }
  }

  if( (location == MII_BMCR) && ((val & BMCR_ANENABLE) != BMCR_ANENABLE) ){
    AVEV3_REG_READ(reg_rxcr, _AVEV3_RXCR);
    AVEV3_REG_READ(reg_txcr_new, _AVEV3_TXCR);
    reg_rxcr_new = reg_rxcr;

    if ( val & BMCR_FULLDPLX ) {
      reg_rxcr_new |= _AVEV3_RXCR_FDUPEN;
      reg_rxcr_new |= _AVEV3_RXCR_FLOCTR;
      reg_txcr_new |= _AVEV3_TXCR_FLOCTR;
    } else {
      reg_rxcr_new &= ~(_AVEV3_RXCR_FDUPEN);
      reg_rxcr_new &= ~(_AVEV3_RXCR_FLOCTR);
      reg_txcr_new &= ~(_AVEV3_TXCR_FLOCTR);
    }


    if ( netif_msg_drv(priv) ) {
      DBG_PRINT("%s: Old rxcr 0x%08lx new_rxcr 0x%08lx new_txcr 0x%08lx \n",
                 dev->name, reg_rxcr, reg_rxcr_new, reg_txcr_new);
    }

    if( reg_rxcr != reg_rxcr_new ){
      AVEV3_REG_WRITE((reg_rxcr_new & ~(_AVEV3_RXCR_RXEN)), _AVEV3_RXCR);

      AVEV3_REG_WRITE( reg_txcr_new, _AVEV3_TXCR);
      AVEV3_REG_WRITE( reg_rxcr_new, _AVEV3_RXCR);
      if ( netif_msg_drv(priv) ) {
        DBG_PRINT("%s: TX RX Register Status Change.\n",dev->name);
      }
    }
  }

  return;
}


unsigned long avev3_get_txfreenum(struct net_device *dev)
{
  struct avev3_private  *priv = netdev_priv(dev);
  unsigned long proc_idx,done_idx,desc_num,free_num;

  if ( netif_msg_drv(priv) ) {
    DBG_PRINT("avev3_get_txfreenum()\n");
  }

  avev3_proc_add("avev3_get_txfreenum");

  proc_idx = priv->tx_proc_idx;
  done_idx = priv->tx_done_idx;
  desc_num = priv->tx_dnum;
  free_num = ((done_idx + desc_num-1) - proc_idx)%desc_num;

  return free_num;
}

int avev3_set_rxdesc(struct net_device *dev, int ring, int entry)
{
  struct avev3_private  *priv = netdev_priv(dev);
  int    retcode = 0;
  struct sk_buff *skb;
  unsigned long reg=0;
  long          align;
  unsigned long descrpt;
  unsigned char *bufptr = NULL; 

  if ( netif_msg_drv(priv) ) {
    DBG_PRINT("avev3_set_rxdesc()\n");
  }

  avev3_proc_add("avev3_set_rxdesc");

  skb = priv->rx_skbs[ring][entry];

  if (skb == NULL) {
    skb = dev_alloc_skb ( _AVEV3_MAX_ETHFRAME + priv->offset + (1024+256) );
    if (skb == NULL) {
      if ( netif_msg_hw(priv) ) {
        printk(KERN_ERR "avev3: Could not allocate skb for rx.\n");
      }
      retcode = -1;
      goto END_PROC;
    }
  } else {
  }

  descrpt = _AVEV3_RXDM + priv->rx_dsa[ring] + entry*_AVEV3_DESC_SIZE;
  reg = (_AVEV3_STS_INTR | _AVEV3_STS_OWN);
  AVEV3_REG_WRITE(reg, descrpt);

  align = (long)skb->tail & 0xff;
  align = 0x100 - align;
  align &= 0xff;
  skb_reserve(skb, (align+512));

  bufptr = skb->tail;

#if defined(CONFIG_ARM)
  reg = virt_to_phys(bufptr);
#else 
  reg = (unsigned long)bufptr;
#endif 

  avev3_purge_cache((unsigned long)bufptr, (unsigned long)(_AVEV3_MAX_ETHFRAME+priv->offset), PURGE_CACHE_D_PURGE_INV);
	
  reg += AVEV3_DDRMEM_SHIFT;  
  wmb();       
  AVEV3_REG_WRITE(reg, descrpt + 4);  

  priv->rx_skbs[ring][entry] = skb;

  reg = (_AVEV3_STS_INTR | _AVEV3_MAX_ETHFRAME);
  wmb();       
  AVEV3_REG_WRITE(reg, descrpt);

END_PROC:
  return retcode;
}


int avev3_refill_rx(struct net_device *dev, int ring)
{
  struct avev3_private  *priv = netdev_priv(dev);
  int entry;

  while ( priv->rx_proc_idx[ring] != priv->rx_done_idx[ring] ) {
    entry = priv->rx_done_idx[ring];
    if( avev3_set_rxdesc(dev, ring, entry) != 0 ){
      break;
    }
    priv->rx_done_idx[ring] = (priv->rx_done_idx[ring] + 1)%(priv->rx_dnum[ring]);
  }

  return 0;
}


int avev3_desc_switch(struct net_device *dev, int flag)
{
  struct avev3_private *priv = netdev_priv(dev);
  unsigned long reg=0;
  int           ring;
  long          loop_cnt = 0;

  if ( netif_msg_drv(priv) ) {
    DBG_PRINT("avev3_desc_switch()\n");
  }
  avev3_proc_add("avev3_desc_switch");

  if( flag == _AVEV3_DESC_START ){
    reg = _AVEV3_DESCC_TD;
    for ( ring = 0; ring < _AVEV3_RXRING_NUM; ring++ ) {
      if( priv->rx_dnum[ring] ){
        reg |= (_AVEV3_DESCC_RD0 << ring);
      }
    }
    AVEV3_REG_WRITE( reg, _AVEV3_DESCC);
  } else if( flag == _AVEV3_DESC_STOP ){
    AVEV3_REG_WRITE( _AVEV3_ZERO, _AVEV3_DESCC);
    loop_cnt = 0;
    while(1){
      udelay(100);
      AVEV3_REG_READ(reg, _AVEV3_DESCC);
      if ( reg == _AVEV3_ZERO ) {
        break;
      }
      loop_cnt++;
      if( loop_cnt > 100) {
        printk(KERN_ERR "%s: Can't stop descriptor (DESCC=0x%08lx).\n", dev->name, reg);
        return -1;
      }
    }
  } else if( flag == _AVEV3_DESC_RX_SUSPEND ){
    AVEV3_REG_READ(reg, _AVEV3_DESCC);
    reg = ( reg | _AVEV3_DESCC_RDSTP) & _AVEV3_DESCC_CTRL_MASK;
    AVEV3_REG_WRITE(reg, _AVEV3_DESCC);
    loop_cnt = 0;
    while(1){
      udelay(100);
      AVEV3_REG_READ(reg, _AVEV3_DESCC);
      if ( reg & (_AVEV3_DESCC_RDSTP<<16) ) {
        break;
      }
      loop_cnt++;
      if( loop_cnt > 1000) {
        printk(KERN_ERR "%s: Can't suspend descriptor (DESCC=0x%08lx).\n", dev->name, reg);
        break;
      }
    }
  } else if( flag == _AVEV3_DESC_RX_PERMIT ){
    AVEV3_REG_READ(reg, _AVEV3_DESCC);
    reg = ( reg & ~_AVEV3_DESCC_RDSTP) & _AVEV3_DESCC_CTRL_MASK;
    AVEV3_REG_WRITE(reg, _AVEV3_DESCC);
  } else {
    return -1;
  }

  return 0;
}

int avev3_descfull_check(struct net_device *dev, int ring)
{
  struct avev3_private *priv = netdev_priv(dev);
  unsigned long addr_rxdcp;
  unsigned long reg_rxdcp1,reg_rxdcp2;
  unsigned long descrpt;
  unsigned long cmdsts;
  int           i;

  if ( netif_msg_drv(priv) ) {
    DBG_PRINT("avev3_descfull_check(ring=%d)\n",ring);
  }
  avev3_proc_add("avev3_descfull_check");

  addr_rxdcp = _AVEV3_RXDCP0 + (ring * 0x4);

  for(i = 0; i < 100; i++){
    AVEV3_REG_READ( reg_rxdcp1, addr_rxdcp);
    descrpt = _AVEV3_RXDM + reg_rxdcp1;
    AVEV3_REG_READ(cmdsts, descrpt);

    AVEV3_REG_READ( reg_rxdcp2, addr_rxdcp);
    if( reg_rxdcp1 == reg_rxdcp2 ){
      break;
    }
  }

  if( cmdsts & _AVEV3_STS_OWN ){
    return 1;
  } else {
    return 0;
  }
}

void avev3_global_reset(struct net_device *dev)
{
  struct avev3_private *priv;
  unsigned long reg_grr,reg_descc;

  if ( dev != NULL ) {
    priv = netdev_priv(dev);
    if ( netif_msg_drv(priv) ) {
      DBG_PRINT("avev3_global_reset()\n");
    }
    AVEV3_REG_READ(reg_grr, _AVEV3_GRR);
    if( (reg_grr & _AVEV3_GRR_GRST) != _AVEV3_GRR_GRST ){
      AVEV3_REG_READ(reg_descc, _AVEV3_DESCC);
      if( (reg_descc & _AVEV3_DESCC_RD0) == _AVEV3_DESCC_RD0 ){
        avev3_desc_switch(dev, _AVEV3_DESC_RX_SUSPEND);
      }
    }
  }
  avev3_proc_add("avev3_global_reset");

  AVEV3_REG_WRITE(_AVEV3_GRR_GRST | _AVEV3_GRR_PHYRST | _AVEV3_GRR_DMACRST, _AVEV3_GRR);

#if 1
  AVEV3_REG_WRITE(_AVEV3_LINKSEL_100M, _AVEV3_LINKSEL);
#endif

  AVEV3_WAIT_FOR_PHY_RESET(avev3_phy_reset_period);


  AVEV3_REG_WRITE(_AVEV3_GRR_GRST, _AVEV3_GRR);

#ifdef AVEV3_ENABLE_LONGWAIT_BY_SLEEP 
  msleep(55);
  avev3_wait_phy_ready();

#else 
#if 1
  mdelay(55);
#endif

#if !defined(AVEV3_SEPARATE_WAIT_FOR_PHY_READY)
  avev3_wait_phy_ready();
#else  
  mdelay(5);  

  avev3_initial_global_reset_completed = 0;
#endif 
#endif 

  AVEV3_REG_WRITE(_AVEV3_ZERO, _AVEV3_GRR);
  udelay(10);  

}

void avev3_wait_phy_ready(void)
{
  AVEV3_WAIT_FOR_PHY_RESET(avev3_phy_reset_stability);

  avev3_initial_global_reset_completed = 1;

}


void avev3_rxf_reset(struct net_device *dev)
{
  struct avev3_private *priv = NULL;
  unsigned long reg_rxcr;
  int           ring;

  priv = netdev_priv(dev);

  if ( netif_msg_rx_err(priv) ) {
    printk(KERN_WARNING "%s: avev3_rxf_reset()\n", dev->name);
  }
  avev3_proc_add("avev3_rxf_reset *****");

  AVEV3_REG_READ(reg_rxcr, _AVEV3_RXCR);
  AVEV3_REG_WRITE((reg_rxcr & ~(_AVEV3_RXCR_RXEN)), _AVEV3_RXCR);

  avev3_desc_switch(dev, _AVEV3_DESC_RX_SUSPEND);

  for ( ring = 0; ring < _AVEV3_RXRING_NUM; ring++ ) {
    avev3_rx(dev,ring, priv->rx_dnum[ring]);
  }

  AVEV3_REG_WRITE(_AVEV3_GRR_RXFFR, _AVEV3_GRR);
  udelay(40);

  AVEV3_REG_WRITE(_AVEV3_ZERO, _AVEV3_GRR);
  udelay(10);

  AVEV3_REG_WRITE(_AVEV3_GI_RXOVF, _AVEV3_GISR);

#if 0
  for ( ring = 0; ring < _AVEV3_RXRING_NUM; ring++ ) {
    priv->rx_proc_idx[ring] = 0;
    priv->rx_done_idx[ring] = 0;
  }
#endif

  avev3_desc_switch(dev, _AVEV3_DESC_RX_PERMIT);

  AVEV3_REG_WRITE(reg_rxcr, _AVEV3_RXCR);

}

int avev3_phy_reset(struct net_device *dev)
{
  struct avev3_private *priv;

  if ( dev != NULL ) {
    priv = netdev_priv(dev);

    if ( netif_msg_hw(priv) ) {
      DBG_PRINT("avev3_phy_reset()\n");
    }

    avev3_proc_add("avev3_phy_reset");
  }

  AVEV3_REG_WRITE(_AVEV3_GRR_PHYRST, _AVEV3_GRR);
  AVEV3_WAIT_FOR_PHY_RESET(avev3_phy_reset_period);
  AVEV3_REG_WRITE(_AVEV3_ZERO, _AVEV3_GRR);
  AVEV3_WAIT_FOR_PHY_RESET(avev3_phy_reset_stability);

  return 0;
}



int avev3_term_ring(struct net_device *dev){
  struct avev3_private *priv = netdev_priv(dev);
  int    ring,entry;

  if ( netif_msg_drv(priv) ) {
    DBG_PRINT("avev3_term_ring()\n");
  }
  avev3_proc_add("avev3_term_ring");

  for ( entry = 0; entry < priv->tx_dnum; entry++ ) {
    if ( priv->tx_skbs[entry] != NULL ) {
      dev_kfree_skb_any( priv->tx_skbs[entry] );
      priv->tx_skbs[entry] = NULL;
    }
  }
  priv->tx_proc_idx = 0;
  priv->tx_done_idx = 0;

  for ( ring = 0; ring < _AVEV3_RXRING_NUM; ring++ ) {
    for ( entry = 0; entry < priv->rx_dnum[ring]; entry++ ) {
      if( priv->rx_skbs[ring][entry] != NULL ){
        dev_kfree_skb_any(priv->rx_skbs[ring][entry]);
        priv->rx_skbs[ring][entry] = NULL;
      }
    }
    priv->rx_proc_idx[ring] = 0;
    priv->rx_done_idx[ring] = 0;
  }

  return 0;
}

int avev3_init_ring(struct net_device *dev)
{
  struct avev3_private *priv = netdev_priv(dev);
  int    retcode = 0;
  int    ring;
  unsigned long rxdsa=0; 

  if ( netif_msg_drv(priv) ) {
    DBG_PRINT("avev3_init_ring()\n");
  }
  avev3_proc_add("avev3_init_ring");

  priv->tx_proc_idx = 0;
  priv->tx_done_idx = 0;
  memset(priv->tx_skbs,0,sizeof(priv->tx_skbs));
  memset(priv->tx_short,0,sizeof(priv->tx_short));

  for ( ring = 0, rxdsa = 0; ring < _AVEV3_RXRING_NUM; ring++ ) {
    priv->rx_proc_idx[ring] = 0;
    priv->rx_done_idx[ring] = 0;
    memset(priv->rx_skbs[ring],0,sizeof(priv->rx_skbs[ring]));

    priv->rx_dsa[ring] = rxdsa;
    rxdsa += priv->rx_dnum[ring] * _AVEV3_DESC_SIZE;
  }

  if( retcode < 0 ){
    avev3_term_ring(dev);
  }
  return retcode;
}


int avev3_emac_init(struct net_device *dev)
{
  struct avev3_private *priv = netdev_priv(dev);
  int ret;

  if ( netif_msg_drv(priv) ) {
    DBG_PRINT("avev3_emac_init()\n");
  }
  avev3_proc_add("avev3_emac_init");

  if ( priv->macaddr_got == 0 ) {
    ret = avev3_get_macaddr(avev3_dev->dev_addr);
    if ( ret < 0 ) {
      printk(KERN_ERR
         "avev3: <FAIL> Taking MacAddr Failed. Default is used.\n");
      avev3_dev->dev_addr[6] = 0x00;    
      avev3_dev->dev_addr[7] = 0x00;    
    } else {
      avev3_dev->dev_addr[6] = 0x00;    
      avev3_dev->dev_addr[7] = 0x00;    
    }
    priv->macaddr_got = 1;
  }

  printk(KERN_INFO "    HWaddr = %02x:%02x:%02x:%02x:%02x:%02x\n",
     avev3_dev->dev_addr[0], avev3_dev->dev_addr[1],
     avev3_dev->dev_addr[2], avev3_dev->dev_addr[3],
     avev3_dev->dev_addr[4], avev3_dev->dev_addr[5]);

  return 0;
}


int avev3_emac_start(struct net_device *dev)
{
  struct avev3_private *priv = netdev_priv(dev);
  unsigned long reg;

  if ( netif_msg_drv(priv) ) {
    DBG_PRINT("avev3_emac_start()\n");
  }
  avev3_proc_add("avev3_emac_start");

  AVEV3_REG_WRITE(*((long *)&dev->dev_addr[0]), _AVEV3_RXMAC1R);
  AVEV3_REG_WRITE(*((long *)&dev->dev_addr[4]), _AVEV3_RXMAC2R);

  reg = _AVEV3_RXCR_RXEN | _AVEV3_RXCR_FDUPEN | _AVEV3_RXCR_DRPEN |
        (_AVEV3_MAX_ETHFRAME & 0x7ff);

  reg |= _AVEV3_RXCR_FLOCTR;

#ifndef AVEV3_MAC_FILTER
  AVEV3_REG_WRITE(reg, _AVEV3_RXCR);
#else
  AVEV3_REG_WRITE(reg | _AVEV3_RXCR_AFEN, _AVEV3_RXCR);
#endif
  AVEV3_REG_WRITE(_AVEV3_TXCR_FLOCTR, _AVEV3_TXCR);

  return 0;
}


int avev3_global_init(struct net_device *dev)
{
  struct avev3_private *priv = netdev_priv(dev);
  unsigned long reg=0;
  unsigned long descrpt;
  int    ring,entry;

  if ( netif_msg_drv(priv) ) {
    DBG_PRINT("avev3_global_init()\n");
  }
  avev3_proc_add("avev3_global_init");

  priv->phy_id = avev3_phy_init(dev);
  priv->mii_if.phy_id = priv->phy_id;

  reg = (priv->tx_dnum * _AVEV3_DESC_SIZE) << 16;
  AVEV3_REG_WRITE( reg, _AVEV3_TXDC);
  for ( entry = 0; entry < priv->tx_dnum; entry++ ) {

    descrpt = _AVEV3_TXDM + entry*_AVEV3_DESC_SIZE;

    AVEV3_REG_WRITE(_AVEV3_ZERO, descrpt);      
    AVEV3_REG_WRITE(_AVEV3_ZERO, descrpt + 4);  
  }

  for ( ring = 0; ring < _AVEV3_RXRING_NUM; ring++ ) {
    reg = ( (priv->rx_dnum[ring] * _AVEV3_DESC_SIZE)<<16 | priv->rx_dsa[ring]);
    AVEV3_REG_WRITE( reg, _AVEV3_RXDC0 + (ring * 0x4));
    for ( entry = 0; entry < priv->rx_dnum[ring]; entry++ ) {
      if( avev3_set_rxdesc(dev, ring, entry) != 0 ){
        break;
      }
    }
  }

  avev3_desc_switch(dev, _AVEV3_DESC_START);

  reg = 0xB0000000; 
  AVEV3_REG_WRITE( reg , _AVEV3_EMCR);
  priv->offset = 2; 
  priv->mii_if.advertising = avev3_mdio_read(dev, priv->phy_id, MII_ADVERTISE);

  avev3_emac_start(dev);

  AVEV3_REG_READ(reg, _AVEV3_IIRQC);
  reg &= 0x0000FFFF; 
  reg = ( reg | (_AVEV3_IIRQC_EN) | (_AVEV3_INTM_COUNT << 16) );
  AVEV3_REG_WRITE(reg, _AVEV3_IIRQC);

  AVEV3_ENABLE_MININT();

#ifdef AVEV3_ENABLE_POL_LINKCHK 
  if(avev3_poll_link != 0){
    AVEV3_REG_READ(reg, _AVEV3_GIMR);
    AVEV3_REG_WRITE((reg & ~(_AVEV3_GI_PHY)), _AVEV3_GIMR);
  }
#endif 

  return 0;
}


#define AVEV3_PROC_WIDTHLIMIT   80
#define AVEV3_PROC_DBG_LINEMAX  40
#define AVEV3_PROC_DBG_WIDTHMAX 32  

static char avev3_proc_msg[AVEV3_PROC_DBG_LINEMAX][AVEV3_PROC_DBG_WIDTHMAX];
static int avev3_proc_cur = 0;  


int avev3_proc_read(char *page, char **start, off_t offset, int count,
           int *eof, void *data)
{
  struct avev3_private *priv = netdev_priv(avev3_dev);
  int i, j, len = 0;
  int limit = count - AVEV3_PROC_WIDTHLIMIT - 60;
#ifdef AVEV3_ENABLE_LONGWAIT_BY_SLEEP 
  unsigned long flags;
#endif 

  if ( netif_msg_drv(priv) ) {
    DBG_PRINT("avev3_proc_read()\n");
  }
#ifdef AVEV3_ENABLE_LONGWAIT_BY_SLEEP 
  spin_lock_irqsave(&priv->lock, flags);
  if(avev3_flg_reset != 0){
    printk("avev3_proc_read reset status \n");
    spin_unlock_irqrestore(&priv->lock, flags);
    return 0;  
  }
#endif 

  i = 0;
  len += sprintf(page +len, "\n----- Start AVEther V3 /proc info -----\n");

  AVEV3_REG_READ(avev3_info[5].value, _AVEV3_BFCR);
  AVEV3_REG_READ(avev3_info[6].value, _AVEV3_GFCR);
  AVEV3_REG_READ(avev3_info[9].value,  _AVEV3_RXFC);
  AVEV3_REG_READ(avev3_info[14].value, _AVEV3_RXOVFFC);

  len += sprintf(page +len, "[AVEV3 Current Infomation:]\n");
  while((avev3_info[i].tag[0] != '\0') && (len < limit)){
    len += sprintf(page + len, "%s:%d\n",
           avev3_info[i].tag, avev3_info[i].value);
    i++;
  }

  len += sprintf(page +len, "\n[AVEV3 Recent Operations:ORDER BY NEW]\n");
  if ( netif_msg_drv(priv) ) {
    DBG_PRINT("current line = %d\n", avev3_proc_cur);
  }
  for(i = 0; (i < AVEV3_PROC_DBG_LINEMAX) && (len < limit); i++){
    j= (avev3_proc_cur - 1 - i + AVEV3_PROC_DBG_LINEMAX)%AVEV3_PROC_DBG_LINEMAX;    
    len += sprintf(page + len, "[%02d]%s\n", j, avev3_proc_msg[j]);
    if(i%5 == 4){
      len += sprintf(page+len, "\n");
    }
  }

  len += sprintf(page +len, "\n-----  End  AVEther V3 /proc info -----\n");

#ifdef AVEV3_ENABLE_LONGWAIT_BY_SLEEP 
  spin_unlock_irqrestore(&priv->lock, flags);
#endif 

  *eof = 1;
  return len;
}


void avev3_proc_add(char *trace_msg)
{
  struct avev3_private *priv;

  if ( avev3_dev != NULL ) {
    priv = netdev_priv(avev3_dev);
    if ( netif_msg_drv(priv) ) {
    }
  } else {
    DBG_PRINT("avev3_proc_add. Line=%d\n", avev3_proc_cur);
  }

  if(debug == 1){
    sprintf(avev3_proc_msg[avev3_proc_cur], trace_msg);
    avev3_proc_cur = (avev3_proc_cur + 1) % AVEV3_PROC_DBG_LINEMAX;
  }

}


void avev3_get_version(struct net_device *dev)
{
  struct avev3_private  *priv = netdev_priv(dev);
  union {
    unsigned long hex;
    char str[4];
  } avev3_id;
  char proc_msg[40];

  if ( netif_msg_drv(priv) ) {
    DBG_PRINT("avev3_get_version()\n");
  }
  avev3_proc_add("avev3_get_version");

  AVEV3_REG_READ(avev3_id.hex, _AVEV3_IDR);
  avev3_endian_change(avev3_id.str);
  memset(priv->id_str, 0, sizeof(priv->id_str));
  memcpy(priv->id_str, avev3_id.str, 4);

  AVEV3_REG_READ(priv->avev3_vr.hex, _AVEV3_VR);

  sprintf(proc_msg, "Panasonic %c%c%c%c(Ver.%04lx)\n",
      priv->id_str[0], priv->id_str[1], priv->id_str[2], priv->id_str[3],
      priv->avev3_vr.hex);
  printk(KERN_INFO "%s", proc_msg);
  printk(KERN_INFO "    Kernel base=0x%08lx, irq=%d\n",
     dev->base_addr, dev->irq);
  avev3_proc_add(proc_msg);
}


struct net_device_stats *avev3_stats(struct net_device *dev)
{
  struct avev3_private *priv = netdev_priv(dev);
  unsigned long reg;
  unsigned long dropped=0;
#ifdef AVEV3_ENABLE_LONGWAIT_BY_SLEEP 
  unsigned long flags;
#endif 


  if ( netif_msg_drv(priv) ) {
    DBG_PRINT("avev3_stats()\n");
  }
  avev3_proc_add("avev3_stats");

#ifdef AVEV3_ENABLE_LONGWAIT_BY_SLEEP 
  spin_lock_irqsave(&priv->lock, flags);
  if(avev3_flg_reset != 0){
    printk("avev3_stats reset status \n");
    spin_unlock_irqrestore(&priv->lock, flags);
    return &priv->stats;  
  }
#endif 


  AVEV3_REG_READ(reg, _AVEV3_BFCR);
  priv->stats.rx_errors = reg;

  AVEV3_REG_READ(reg, _AVEV3_RXOVFFC);
  dropped += reg;
  priv->stats.rx_dropped = dropped; 

#ifdef AVEV3_ENABLE_LONGWAIT_BY_SLEEP 
  spin_unlock_irqrestore(&priv->lock, flags);
#endif 

  return &priv->stats;
}


int avev3_ioctl(struct net_device *dev, struct ifreq *req, int cmd)
{
  struct avev3_private *priv = netdev_priv(dev);
  struct mii_ioctl_data *mii_ctl = if_mii(req);
  unsigned long flags;
  u32 *data = (u32 *) & req->ifr_data;


    
  if ( netif_msg_drv(priv) ) {
    DBG_PRINT("avev3_ioctl()\n");
  }
  avev3_proc_add("avev3_ioctl");



  spin_lock_irqsave(&priv->lock, flags);

#ifdef AVEV3_ENABLE_LONGWAIT_BY_SLEEP 
  if(avev3_flg_reset != 0){
    printk("avev3_ioctl reset status \n");
    spin_unlock_irqrestore(&priv->lock, flags);
    return -EAGAIN;  
  }
#endif 

  switch(cmd) {


  case SIOCDEVPRIVATE:
    *data = mii_link_ok(&priv->mii_if);
    spin_unlock_irqrestore(&priv->lock, flags);
    return 0;
  case SIOCGMIIPHY:     
    (if_mii(req))->phy_id = priv->phy_id;
  case SIOCGMIIREG:     
    mii_ctl->val_out = avev3_mdio_read(dev, mii_ctl->phy_id &
                      priv->mii_if.phy_id_mask,
                      mii_ctl->reg_num &
                      priv->mii_if.reg_num_mask);
    spin_unlock_irqrestore(&priv->lock, flags);
    return 0;
  case SIOCSMIIREG:     
    if (!capable(CAP_NET_ADMIN)) {
      spin_unlock_irqrestore(&priv->lock, flags);
      return -EPERM;
    }
    avev3_mdio_write(dev, mii_ctl->phy_id, mii_ctl->reg_num, mii_ctl->val_in);
    spin_unlock_irqrestore(&priv->lock, flags);
    return 0;
  default:
    spin_unlock_irqrestore(&priv->lock, flags);
    return -EOPNOTSUPP;
  }

  return(0);
}


void avev3_multicast(struct net_device *dev)
{
  struct avev3_private *priv = netdev_priv(dev);
  unsigned long flags;
  unsigned long reg;

  if (netif_msg_rx_status(priv)) {
    DBG_PRINT("%s: avev3_multicast %04x -> %04x\n",
          dev->name, priv->flags, dev->flags);
  }
  avev3_proc_add("avev3_multicast");


  priv->flags = dev->flags;

  spin_lock_irqsave(&priv->lock, flags);

#ifdef AVEV3_ENABLE_LONGWAIT_BY_SLEEP 
  if(avev3_flg_reset != 0){
    printk("avev3_multicast reset status \n");
    spin_unlock_irqrestore(&priv->lock, flags);
    return;  
  }
#endif 

  if (dev->flags & IFF_PROMISC) {
    if (netif_msg_rx_status(priv)) {
      DBG_PRINT("Promiscas Mode\n");
    }
    AVEV3_REG_READ(reg, _AVEV3_RXCR);
    AVEV3_REG_WRITE((reg & ~(_AVEV3_RXCR_AFEN)), _AVEV3_RXCR);
  } else {
    if (netif_msg_rx_status(priv)) {
      DBG_PRINT("non-Promiscas Mode\n");
    }
    AVEV3_REG_READ(reg, _AVEV3_RXCR);
    AVEV3_REG_WRITE((reg | _AVEV3_RXCR_AFEN), _AVEV3_RXCR);
  }


  spin_unlock_irqrestore(&priv->lock, flags);

}


void avev3_util_linkchk_needspinlock(struct net_device *dev)
{
  struct avev3_private  *priv = netdev_priv(dev);
  int                   link_status_change = _AVEV3_LS_DOWN;
  unsigned long         partner, bmcr, result, pause_cap;
  unsigned long         reg, reg_txcr, reg_rxcr;


  if(mii_link_ok(&priv->mii_if) != 0){
    link_status_change = _AVEV3_LS_UP;
  }

  if( link_status_change == _AVEV3_LS_UP ) {


    if ( netif_msg_ifup(priv) ) {
      DBG_PRINT("Link has gone up.\n");
    }
    partner = avev3_mdio_read(dev, priv->phy_id, MII_LPA);
    if ( netif_msg_link(priv) ) {
      DBG_PRINT("%s: Old partner %04x, new %08lx, adv %08x.\n",
                dev->name, priv->partner, partner, priv->mii_if.advertising);
    }

    if (partner != (unsigned long)(priv->partner)) {
      if (netif_msg_link(priv)) {
        DBG_PRINT("%s: Link status change.\n", dev->name);
      }
      priv->partner = partner;
    }

    mii_check_link(&priv->mii_if);

    bmcr = avev3_mdio_read(dev, priv->phy_id, MII_BMCR);
    if ( bmcr & BMCR_ANENABLE ) {
      result = mii_nway_result(priv->mii_if.advertising & partner);
      if ( result == LPA_100FULL || result == LPA_10FULL ) {
        priv->mii_if.full_duplex = 1;
        if ( netif_msg_link(priv) ) {
          DBG_PRINT("Full Duplex by Auto-Negotiation in avev3_util_linkchk_needspinlock()\n");
        }
        if( (priv->mii_if.advertising & partner) & _BIT10 ){ 
          pause_cap = 1;
        } else {
          pause_cap = 0;
        }
      } else {
        priv->mii_if.full_duplex = 0;
        if ( netif_msg_link(priv) ) {
          DBG_PRINT("Half Duplex by Auto-Negotiation in avev3_util_linkchk_needspinlock()\n");
        }
        pause_cap = 0;
      }

#ifdef AVEV3_ENABLE_LINKSPEED_SET 
      AVEV3_REG_READ(reg, _AVEV3_LINKSEL);
      if ( (result == LPA_100FULL) || (result == LPA_100HALF) ) {
        reg |= _AVEV3_LINKSEL_100M;       
      }
      else {
        reg &= ~(_AVEV3_LINKSEL_100M);    
      }
      AVEV3_REG_WRITE( reg, _AVEV3_LINKSEL);
#endif 

    } else {
      if ( bmcr & BMCR_FULLDPLX ) {
        if ( netif_msg_link(priv) ) {
          DBG_PRINT("Full Duplex by Manual in avev3_util_linkchk_needspinlock(). duplex=%d\n",
                    priv->mii_if.full_duplex);
        }
        pause_cap = 1;
      } else {
        if ( netif_msg_link(priv) ) {
          DBG_PRINT("Half Duplex by Manual in avev3_util_linkchk_needspinlock(). duplex=%d\n",
                    priv->mii_if.full_duplex);
        }
        pause_cap = 0;
      }

#ifdef AVEV3_ENABLE_LINKSPEED_SET 
      AVEV3_REG_READ(reg, _AVEV3_LINKSEL);
      if (bmcr & BMCR_SPEED100) {
        reg |= _AVEV3_LINKSEL_100M;       
      }
      else {
        reg &= ~(_AVEV3_LINKSEL_100M);    
      }
      AVEV3_REG_WRITE( reg, _AVEV3_LINKSEL);
#endif 
    }

    AVEV3_REG_READ(reg, _AVEV3_RXCR);
    AVEV3_REG_READ(reg_txcr, _AVEV3_TXCR);
    reg_rxcr = reg;
    if( priv->mii_if.full_duplex ){
      reg_rxcr |= _AVEV3_RXCR_FDUPEN;
    } else {
      reg_rxcr &= ~(_AVEV3_RXCR_FDUPEN);
    }
    if( pause_cap ){
      reg_rxcr |= _AVEV3_RXCR_FLOCTR;
      reg_txcr |= _AVEV3_TXCR_FLOCTR;
    } else {
      reg_rxcr &= ~(_AVEV3_RXCR_FLOCTR);
      reg_txcr &= ~(_AVEV3_TXCR_FLOCTR);
    }

    if( reg != reg_rxcr ){
      if ( netif_msg_link(priv) ) {
        DBG_PRINT("%s: Previous MAC mode(rxcr=0x%08lx).\n", dev->name, reg);
        DBG_PRINT("%s: Change MAC mode(txcr=0x%08lx  rxcr=0x%08lx).\n",
                   dev->name, reg_txcr, reg_rxcr);
      }
      AVEV3_REG_WRITE((reg_rxcr & ~(_AVEV3_RXCR_RXEN)), _AVEV3_RXCR);

      AVEV3_REG_WRITE( reg_txcr, _AVEV3_TXCR);
      AVEV3_REG_WRITE( reg_rxcr, _AVEV3_RXCR);
    }

  } else if ( link_status_change == _AVEV3_LS_DOWN ) {
    if ( netif_msg_ifdown(priv) ) {
      DBG_PRINT("Link has gone down.\n");
    }

    mii_check_link(&priv->mii_if);

  } else {
    if (netif_msg_intr(priv)) {
      printk(KERN_WARNING "avev3: Link status changed, " \
             "but we don't know what it is.\n");
    }
  }

  return;
}



void avev3_timer(unsigned long data)
{
  struct net_device *dev = (struct net_device *)data;
  struct avev3_private  *priv = netdev_priv(dev);
  unsigned long flags;

  if ( netif_msg_timer(priv) ) {
    DBG_PRINT("avev3_timer()\n");
  }

  spin_lock_irqsave(&priv->lock, flags);

#ifdef AVEV3_ENABLE_LONGWAIT_BY_SLEEP 
  if(avev3_flg_reset != 0){
    printk("avev3_timer reset status \n");
    spin_unlock_irqrestore(&priv->lock, flags);
    return;  
  }
#endif 


  avev3_tx_completion(dev);


#ifdef AVEV3_ENABLE_POL_LINKCHK 
  if(avev3_poll_link != 0){
    avev3_util_linkchk_needspinlock(dev);
  }
#endif 

  priv->timer.expires = RUN_AT(_AVEV3_MNTR_TIME);
  add_timer(&priv->timer);

  spin_unlock_irqrestore(&priv->lock, flags);
}


int avev3_tx_completion(struct net_device *dev)
{
  struct avev3_private  *priv = netdev_priv(dev);
  unsigned long reg;
  unsigned long proc_idx,done_idx,desc_num;
  unsigned long freebuf_flg = 0;  

  if ( netif_msg_tx_done(priv) ) {
    DBG_PRINT("avev3_tx_completion()\n");
  }
  avev3_proc_add("avev3_tx_completion");

  AVEV3_REG_WRITE(_AVEV3_GI_TX, _AVEV3_GISR);

  proc_idx = priv->tx_proc_idx;
  done_idx = priv->tx_done_idx;
  desc_num = priv->tx_dnum;

  while( proc_idx != done_idx ){

    AVEV3_REG_READ(reg, _AVEV3_TXDM + (done_idx*_AVEV3_DESC_SIZE));
    if( reg & _AVEV3_STS_OWN ){
      break; 
    } else {
      if ( reg & _AVEV3_STS_OK ){
        priv->stats.tx_packets++;
        priv->stats.tx_bytes += reg & _AVEV3_STS_PKTLEN;
        avev3_info[7].value  += reg & _AVEV3_STS_PKTLEN;
      } else {
        if (netif_msg_tx_err(priv)) {
          DBG_PRINT("%s: Tx Error. status %08lx.\n",dev->name, reg);
        }
        priv->stats.tx_errors++;
        if ( reg & (_AVEV3_STS_OWC | _AVEV3_STS_EC) ){
          priv->stats.collisions++;
        }
      }

      if ( priv->tx_skbs[done_idx] != NULL ) {
        dev_kfree_skb_any( priv->tx_skbs[done_idx] );
        priv->tx_skbs[done_idx] = NULL;
        freebuf_flg = 1;
      }
      done_idx = (done_idx+1)%desc_num;
    }
  }

  priv->tx_done_idx = done_idx;

  if ( netif_queue_stopped(dev) ) {
    if( netif_running(dev) ){   
      if( freebuf_flg != 0 ){
        netif_wake_queue(dev);
      }
    }
  }

  return 0;
}


int avev3_start_xmit(struct sk_buff *skb, struct net_device *dev)
{
  struct avev3_private  *priv = netdev_priv(dev);
  unsigned long flags;
  int   i;
  int   txlen;
  unsigned long reg;
  unsigned long proc_idx,done_idx,desc_num,free_num;
  unsigned long descrpt;
  unsigned char *bufptr = NULL; 
  int   offset = 0;

  if ( netif_msg_tx_queued(priv) ) {
    DBG_PRINT("avev3_start_xmit()\n");
  }
  avev3_proc_add("avev3_start_xmit");



  spin_lock_irqsave(&priv->lock, flags);

#ifdef AVEV3_ENABLE_LONGWAIT_BY_SLEEP 
  if(avev3_flg_reset != 0){
    printk("avev3_start_xmit reset status \n");
    dev_kfree_skb_any(skb);
    spin_unlock_irqrestore(&priv->lock, flags);
    return NETDEV_TX_OK;  
  }
#endif 


  if (skb->len > ETH_FRAME_LEN) {
    dev_kfree_skb_any(skb);
    spin_unlock_irqrestore(&priv->lock, flags);
    return NETDEV_TX_OK;
  }


  free_num = avev3_get_txfreenum(dev);
  if( (free_num < 2) || (priv->tx_dnum - free_num > _AVEV3_TXFREE_THRESHOLD) ){
    avev3_tx_completion(dev);
    free_num = avev3_get_txfreenum(dev);
    if( free_num < 2 ){
      if ( netif_msg_tx_queued(priv) ) {
        DBG_PRINT("%s: stop_queue : there is no tx free entry num(%ld).\n", dev->name,free_num);
      }
      netif_stop_queue(dev);
    }
  } else {
  }

  if(free_num < 1){
    spin_unlock_irqrestore(&priv->lock, flags);
    return (NETDEV_TX_BUSY);
  }


  proc_idx = priv->tx_proc_idx;
  done_idx = priv->tx_done_idx;
  desc_num = priv->tx_dnum;

  priv->tx_skbs[proc_idx] = skb;
  if ( netif_msg_tx_queued(priv) ) {
    DBG_PRINT("%s: tx proc_idx = %d, done_idx = %d, desc_num = %d, free_num = %d, skb->len = 0x%08X\n",
               dev->name, (int)proc_idx, (int)done_idx, (int)desc_num, (int)free_num, (int)skb->len);
  }

  offset = priv->offset;

  if( skb->len < ETH_ZLEN ) {   
    memset((priv->tx_short[proc_idx] + offset + skb->len), 0x00, (ETH_ZLEN - skb->len));
    memcpy(priv->tx_short[proc_idx]+offset, skb->data, skb->len);
    bufptr = priv->tx_short[proc_idx];
    txlen = ETH_ZLEN;
  } else {
    bufptr = skb->data-offset;
    txlen = skb->len;
  }

  if ( netif_msg_pktdata(priv) ) {
    unsigned long *pbuf;
    unsigned long work_data;

    DBG_PRINT("buff address = 0x%08lx\n",(unsigned long)bufptr);
    pbuf = (unsigned long *)(bufptr);
    for(i = 0; i < (txlen + offset) / 4; i++){
      work_data = *(pbuf + i);
      avev3_endian_change(&work_data);
      DBG_PRINT("tx[%03d] = 0x%08X\n", i, (int)work_data);
    }
    if((txlen + offset) % 4){
      work_data = *(pbuf + i);
      avev3_endian_change(&work_data);
      DBG_PRINT("tx[%03d] = 0x%08X\n", i, (int)work_data);
    }
  }

  descrpt = _AVEV3_TXDM + (proc_idx*_AVEV3_DESC_SIZE);

#if defined(CONFIG_ARM)
  reg = virt_to_phys(bufptr);
#else 
  reg = (unsigned long)bufptr;
#endif 

  avev3_purge_cache((unsigned long)bufptr, (unsigned long)(txlen+offset), PURGE_CACHE_D_PURGE_INV);

  reg += AVEV3_DDRMEM_SHIFT;  
  AVEV3_REG_WRITE(reg, descrpt + 4);

  reg = ( _AVEV3_STS_OWN | txlen );
  if ( netif_queue_stopped(dev) ){
    reg |= _AVEV3_STS_INTR;
  }

  wmb();       
  AVEV3_REG_WRITE(reg, descrpt); 

  priv->tx_proc_idx = (proc_idx + 1)%desc_num;

  spin_unlock_irqrestore(&priv->lock, flags);

  dev->trans_start = jiffies;

  return NETDEV_TX_OK;

}


int avev3_rx(struct net_device *dev, int ring, int num)
{
  struct avev3_private  *priv = netdev_priv(dev);
  int ret;
  unsigned long cmdsts;
  int   pkt_len;
  struct sk_buff *skb;
  int i;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,32) 
  struct ethhdr *eh;
  unsigned char ih_proto = 0;
#endif 


  unsigned long descrpt;

  unsigned long proc_idx,done_idx,desc_num;
  int   restcnt;



  if (netif_msg_intr(priv)) {
    DBG_PRINT("%s: avev3_rx().\n", dev->name);
    avev3_proc_add("avev3_rx()");
  }

  proc_idx = priv->rx_proc_idx[ring];
  done_idx = priv->rx_done_idx[ring];
  desc_num = priv->rx_dnum[ring];
  restcnt  = ((done_idx + desc_num-1) - proc_idx)%desc_num;
  for(ret=0; ret<num; ret++ ) {


    if (--restcnt < 0)
      break;

    descrpt = _AVEV3_RXDM + priv->rx_dsa[ring] + proc_idx*_AVEV3_DESC_SIZE;

    AVEV3_REG_READ(cmdsts, descrpt);

    if( (cmdsts & _AVEV3_STS_OWN) != _AVEV3_STS_OWN ){
      break;
    }

    if ( (cmdsts & _AVEV3_STS_OK) != _AVEV3_STS_OK ) {
      priv->stats.rx_errors++;
      if ( netif_msg_rx_err(priv) ) {
        printk(KERN_ERR "%s;  Recv error at (proc_idx=%ld) of ring(%d)\n",
                        dev->name,proc_idx,ring);
      }
      proc_idx = (proc_idx+1)%desc_num;
      continue;
    }

    pkt_len = cmdsts & _AVEV3_STS_PKTLEN;
    if (netif_msg_rx_status(priv)) {
      DBG_PRINT("%s:  avev3_rx() cmdsts %08lx len %d.\n",
        dev->name, cmdsts, pkt_len);
    }

    if ( pkt_len < 0 ) {
      printk(KERN_ERR "%s: 0 length packet received???\n",dev->name);
      break;
    }

    skb = priv->rx_skbs[ring][proc_idx];
    priv->rx_skbs[ring][proc_idx] = NULL;
    skb->dev = dev;

#if defined(CONFIG_ARM)
    avev3_purge_cache((unsigned long)(skb->data), (unsigned long)(pkt_len+2), PURGE_CACHE_D_INV);
#endif 

    skb_reserve(skb, 2);        
    skb_put(skb, (unsigned int)pkt_len);

    if ( netif_msg_pktdata(priv) ) {
      unsigned long *pbuf;
      unsigned long work_data;

      DBG_PRINT("  skb_len = 0x%08X\n", (int)skb->len);
      DBG_PRINT("   (The top 2 bytes are offset.)\n");
      pbuf = (unsigned long *)(skb->data - 2);
      for(i = 0; i < (skb->len + 2) / 4; i++){
        work_data = *(pbuf + i);
        avev3_endian_change(&work_data);
        DBG_PRINT("   pbuf[%03d] = 0x%08X\n", i, (int)work_data);
      }
      if((skb->len + 2) % 4){
        work_data = *(pbuf + i);
        avev3_endian_change(&work_data);
        DBG_PRINT("   pbuf[%03d] = 0x%08X\n", i, (int)work_data);
      }
    }

    if ( cmdsts & _AVEV3_STS_VLAN ) {
#define        VLAN_OFFSET        4
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,32) 
      skb->mac.raw = skb->data;
      eh = eth_hdr(skb);
      if ( ntohs(eh->h_proto) !=  ETH_P_8021Q ) {
        if ( netif_msg_rx_err(priv) ) {
          DBG_PRINT("VLAN but different Protocol ?\n");
        }
      }
#endif 
      avev3_info[4].value++;
      for ( i = ETH_HLEN-3; i >= 0; i-- ) {
        *(skb->data + VLAN_OFFSET + i) = *(skb->data + i);
      }
      skb_pull(skb, VLAN_OFFSET);
    }
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,32) 
    skb->mac.raw = skb->data;
    eh = eth_hdr(skb);

    if ( ntohs(eh->h_proto) == ETH_P_IP ) {
      skb->nh.iph = (struct iphdr *)(skb->data + ETH_HLEN);
      ih_proto = skb->nh.iph->protocol;
    } else if ( ntohs(eh->h_proto) == ETH_P_IPV6 ) {
      skb->nh.ipv6h = (struct ipv6hdr *)(skb->data + ETH_HLEN);
      ih_proto = skb->nh.ipv6h->nexthdr;
    }
#endif 

    skb->protocol = eth_type_trans(skb, dev);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,32) 
    if ( skb->protocol != eh->h_proto ) {
      if ( netif_msg_rx_err(priv) ) {
        DBG_PRINT("Ether Protocol Analysis Error.\n");
      }
    }
#endif 



    avev3_info[8].value += pkt_len;
    priv->stats.rx_packets++;
    priv->stats.rx_bytes += pkt_len;

    dev->last_rx = jiffies;
    netif_rx(skb);

    proc_idx = (proc_idx+1)%desc_num;
  }

  priv->rx_proc_idx[ring] = proc_idx;
  avev3_refill_rx(dev,ring);

  return ret;
}


#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,32) 
irqreturn_t avev3_interrupt(int irq, void *dev_id, struct pt_regs *regs)
#else 
irqreturn_t avev3_interrupt(int irq, void *dev_id)
#endif 

{
  unsigned long     flags;  
  unsigned long     gimr;   
  unsigned long     gisr;   
  unsigned long     partner, bmcr, result, pause_cap;
  unsigned long     reg,reg_txcr,reg_rxcr;
  struct avev3_private  *priv;
  struct net_device *dev = (struct net_device *)dev_id;
  int link_status_change;


#ifdef _AVEV3_DEBUG_INTR
  printk(KERN_INFO "avev3_interrupt in\n");
#endif

  if(!dev_id) {
    return IRQ_NONE;
  }

  priv = netdev_priv(dev);

  if (netif_msg_intr(priv)) {
    DBG_PRINT("%s: avev3_interrupt().\n", dev->name);
  }


  spin_lock_irqsave(&(priv->lock), flags);

#ifdef AVEV3_ENABLE_LONGWAIT_BY_SLEEP 
  if(avev3_flg_reset != 0){
    printk("avev3_interrupt reset status \n");
    spin_unlock_irqrestore(&(priv->lock), flags);
    return IRQ_NONE;  
  }
#endif 

  AVEV3_REG_READ(gimr, _AVEV3_GIMR);
  if (netif_msg_intr(priv)) {
    DBG_PRINT("%s:  int_mask = 0x%08lX\n", dev->name, gimr);
  }

  AVEV3_REG_WRITE(_AVEV3_ZERO, _AVEV3_GIMR);

  AVEV3_REG_READ(gisr, _AVEV3_GISR);
  if (netif_msg_intr(priv)) {
    DBG_PRINT("%s:  int_status = 0x%08lX(mask:0x%08lX)\n", dev->name,
          gisr, gimr);
  }

  if ( gisr & _AVEV3_GI_DMAC ) {
    if ( netif_msg_hw(priv) ) {
      printk(KERN_WARNING
             "avev3: AVEV3 DMAC Error. Currently do nothing.\n");
    }
    AVEV3_REG_WRITE(_AVEV3_GI_DMAC, _AVEV3_GISR);
  }

  if ( gisr & _AVEV3_GI_RXERR ) {
    if ( netif_msg_rx_err(priv) ) {
      printk(KERN_ERR "avev3: Recieve packet more than buffsize. Currently do nothing.\n");
    }
    AVEV3_REG_WRITE(_AVEV3_GI_RXERR, _AVEV3_GISR);
  }
  if ( (gisr & gimr) == 0 ) {
    if ( netif_msg_hw(priv) ) { 
      DBG_PRINT("Interrupt by Masked Factors!\n");
    }
    goto END_PROC;
  }

  if ( (gisr & gimr) & _AVEV3_GI_RXOVF ) {
    if ( netif_msg_rx_err(priv) ) {
      printk(KERN_WARNING "avev3: AVEV3 RxFIFO Overflow.\n");
    }
    avev3_info[0].value++;
    priv->stats.rx_fifo_errors++;
  #ifndef RXFIFO_RESET_DISABLE
    avev3_rxf_reset(dev);
    goto END_PROC;  
  #else 
    printk(KERN_ERR "----------------------------------- \n");
    printk(KERN_ERR "avev3: AVEV3 RxFIFO Overflow occurred.\n");
    printk(KERN_ERR "But we do nothing.\n");
    printk(KERN_ERR "----------------------------------- \n");
    avev3_panic(2);
  #endif 
  }

  if ( (gisr & gimr) & _AVEV3_GI_RXDROP ) {
    if ( netif_msg_rx_status(priv) ) {
      printk(KERN_WARNING "avev3: AVEV3 Rx Drop.\n");
    }
    priv->stats.rx_dropped++;
    AVEV3_REG_WRITE(_AVEV3_GI_RXDROP, _AVEV3_GISR);
  }

  if ( (gisr & gimr) & _AVEV3_GI_RXIINT ) {
    avev3_rx(dev, 0, _AVEV3_IIRQ_RCVCNT);
    AVEV3_REG_WRITE(_AVEV3_GI_RXIINT, _AVEV3_GISR);
  }

  if ( (gisr & gimr) & _AVEV3_GI_TX ) { 
    avev3_tx_completion(dev);
  }

  if ( (gisr & gimr) & _AVEV3_GI_PHY ) {    
    if ( netif_msg_intr(priv) ) {
      DBG_PRINT("avev3_interrupt phy\n");
    }

    link_status_change = avev3_phy_intr_chkstate_negate(dev);
    AVEV3_REG_WRITE(_AVEV3_GI_PHY, _AVEV3_GISR);

    if( link_status_change == _AVEV3_LS_UP ) {
      if ( netif_msg_ifup(priv) ) {
        DBG_PRINT("Link has gone up.\n");
      }
      partner = avev3_mdio_read(dev, priv->phy_id, MII_LPA);
      if ( netif_msg_link(priv) ) {
        DBG_PRINT("%s: Old partner %04x, new %08lx, adv %08x.\n",
                  dev->name, priv->partner, partner, priv->mii_if.advertising);
      }
      if (partner != (unsigned long)(priv->partner)) {
        if (netif_msg_link(priv)) {
          DBG_PRINT("%s: Link status change.\n", dev->name);
        }
        priv->partner = partner;
      }

      mii_check_link(&priv->mii_if);

      bmcr = avev3_mdio_read(dev, priv->phy_id, MII_BMCR);
      if ( bmcr & BMCR_ANENABLE ) {
        result = mii_nway_result(priv->mii_if.advertising & partner);
        if ( result == LPA_100FULL || result == LPA_10FULL ) {
          priv->mii_if.full_duplex = 1;
          if ( netif_msg_link(priv) ) {
            DBG_PRINT("Full Duplex by Auto-Negotiation in avev3_interrupt()\n");
          }
          if( (priv->mii_if.advertising & partner) & _BIT10 ){ 
            pause_cap = 1;
          } else {
            pause_cap = 0;
          }
        } else {
          priv->mii_if.full_duplex = 0;
          if ( netif_msg_link(priv) ) {
            DBG_PRINT("Half Duplex by Auto-Negotiation in avev3_interrupt()\n");
          }
          pause_cap = 0;
        }

#ifdef AVEV3_ENABLE_LINKSPEED_SET 
        AVEV3_REG_READ(reg, _AVEV3_LINKSEL);
        if ( (result == LPA_100FULL) || (result == LPA_100HALF) ) {
          reg |= _AVEV3_LINKSEL_100M;       
        }
        else {
          reg &= ~(_AVEV3_LINKSEL_100M);    
        }
        AVEV3_REG_WRITE( reg, _AVEV3_LINKSEL);
#endif 

      } else {
        if ( bmcr & BMCR_FULLDPLX ) {
          if ( netif_msg_link(priv) ) {
            DBG_PRINT("Full Duplex by Manual in avev3_interrupt(). duplex=%d\n",
                      priv->mii_if.full_duplex);
          }
          pause_cap = 1;
        } else {
          if ( netif_msg_link(priv) ) {
            DBG_PRINT("Half Duplex by Manual in avev3_interrupt(). duplex=%d\n",
                      priv->mii_if.full_duplex);
          }
          pause_cap = 0;
        }

#ifdef AVEV3_ENABLE_LINKSPEED_SET 
        AVEV3_REG_READ(reg, _AVEV3_LINKSEL);
        if (bmcr & BMCR_SPEED100) {
          reg |= _AVEV3_LINKSEL_100M;       
        }
        else {
          reg &= ~(_AVEV3_LINKSEL_100M);    
        }
        AVEV3_REG_WRITE( reg, _AVEV3_LINKSEL);
#endif 

      }

      AVEV3_REG_READ(reg, _AVEV3_RXCR);
      AVEV3_REG_READ(reg_txcr, _AVEV3_TXCR);
      reg_rxcr = reg;
      if( priv->mii_if.full_duplex ){
        reg_rxcr |= _AVEV3_RXCR_FDUPEN;
      } else {
        reg_rxcr &= ~(_AVEV3_RXCR_FDUPEN);
      }
      if( pause_cap ){
        reg_rxcr |= _AVEV3_RXCR_FLOCTR;
        reg_txcr |= _AVEV3_TXCR_FLOCTR;
      } else {
        reg_rxcr &= ~(_AVEV3_RXCR_FLOCTR);
        reg_txcr &= ~(_AVEV3_TXCR_FLOCTR);
      }

      if( reg != reg_rxcr ){
        if ( netif_msg_link(priv) ) {
          DBG_PRINT("%s: Previous MAC mode(rxcr=0x%08lx).\n", dev->name, reg);
          DBG_PRINT("%s: Change MAC mode(txcr=0x%08lx  rxcr=0x%08lx).\n",
                     dev->name, reg_txcr, reg_rxcr);
        }
        AVEV3_REG_WRITE((reg_rxcr & ~(_AVEV3_RXCR_RXEN)), _AVEV3_RXCR);

        AVEV3_REG_WRITE( reg_txcr, _AVEV3_TXCR);
        AVEV3_REG_WRITE( reg_rxcr, _AVEV3_RXCR);
      }

    } else if ( link_status_change == _AVEV3_LS_DOWN ) {
      if ( netif_msg_ifdown(priv) ) {
        DBG_PRINT("Link has gone down.\n");
      }

      mii_check_link(&priv->mii_if);

    } else {
      if (netif_msg_intr(priv)) {
        printk(KERN_WARNING "avev3: Link status changed, " \
               "but we don't know what it is.\n");
      }
    }

  }

  if ( netif_msg_intr(priv) ) {
    AVEV3_REG_READ(gisr, _AVEV3_GISR);
    if ( gisr & gimr ) {
      DBG_PRINT("interrupt again... perhaps waken up soon. (0x%08lx)\n",
        gisr);
    }
  }

 END_PROC:


  AVEV3_REG_WRITE(gimr, _AVEV3_GIMR);

  avev3_peaks_clear_irq(dev->irq);

  spin_unlock_irqrestore(&(priv->lock), flags);

  if (netif_msg_intr(priv)) {
    AVEV3_REG_READ(gisr, _AVEV3_GISR);
    DBG_PRINT("%s: exiting interrupt, status=%08lx.\n",
          dev->name, gisr);
  }

  if (netif_msg_intr(priv)) {
    printk(KERN_INFO "avev3_interrupt out\n");
  }

  return IRQ_HANDLED;
}


static void avev3_get_drvinfo(struct net_device *dev,
                 struct ethtool_drvinfo *info)
{
  strncpy(info->driver, "AVEV3", sizeof(info->driver)-1);
  strncpy(info->version, drv_version, sizeof(info->version)-1);
}

static int avev3_get_settings(struct net_device *dev, struct ethtool_cmd *ecmd)
{
  struct avev3_private *priv = netdev_priv(dev);
  unsigned long flags;


  spin_lock_irqsave(&priv->lock, flags);

#ifdef AVEV3_ENABLE_LONGWAIT_BY_SLEEP 
  if(avev3_flg_reset != 0){
    printk("avev3_get_settings reset status \n");
    spin_unlock_irqrestore(&priv->lock, flags);
    return -EAGAIN;  
  }
#endif 

  mii_ethtool_gset(&priv->mii_if, ecmd);
  spin_unlock_irqrestore(&priv->lock, flags);
  return 0;
}

static int avev3_set_settings(struct net_device *dev, struct ethtool_cmd *ecmd)
{
  struct avev3_private *priv = netdev_priv(dev);
  int res;
  unsigned long flags;


  spin_lock_irqsave(&priv->lock, flags);

#ifdef AVEV3_ENABLE_LONGWAIT_BY_SLEEP 
  if(avev3_flg_reset != 0){
    printk("avev3_set_settings reset status \n");
    spin_unlock_irqrestore(&priv->lock, flags);
    return -EAGAIN;  
  }
#endif 

  res = mii_ethtool_sset(&priv->mii_if, ecmd);
  spin_unlock_irqrestore(&priv->lock, flags);
  return res;
}

static int avev3_nway_reset(struct net_device *dev)
{
  struct avev3_private *priv = netdev_priv(dev);
  int res;
  unsigned long flags;
  

  spin_lock_irqsave(&priv->lock, flags);

#ifdef AVEV3_ENABLE_LONGWAIT_BY_SLEEP 
  if(avev3_flg_reset != 0){
    printk("avev3_nway_reset reset status \n");
    spin_unlock_irqrestore(&priv->lock, flags);
    return 0;  
  }
#endif 

  res = mii_nway_restart(&priv->mii_if);
  spin_unlock_irqrestore(&priv->lock, flags);
  return res;
}

static u32 avev3_get_link(struct net_device *dev)
{
  struct avev3_private *priv = netdev_priv(dev);
  int res;
  unsigned long flags;
  
  spin_lock_irqsave(&priv->lock, flags);

#ifdef AVEV3_ENABLE_LONGWAIT_BY_SLEEP 
  if(avev3_flg_reset != 0){
    printk("avev3_get_link reset status \n");
    spin_unlock_irqrestore(&priv->lock, flags);
    return 0;  
  }
#endif 

  res = mii_link_ok(&priv->mii_if);
  spin_unlock_irqrestore(&priv->lock, flags);
  return res;
}

static u32 avev3_get_msglevel(struct net_device *dev)
{
  struct avev3_private *priv = netdev_priv(dev);
  return priv->msg_enable;
}

static void avev3_set_msglevel(struct net_device *dev, u32 value)
{
  struct avev3_private *priv = netdev_priv(dev);
  priv->msg_enable = value;
  return;
}

static u32 avev3_get_rx_csum(struct net_device *dev)
{
  return 0;
}

static u32 avev3_get_tx_csum(struct net_device *dev)
{
  return 0;
}

static struct ethtool_ops ethtool_ops = {
  .get_drvinfo = avev3_get_drvinfo,
  .get_settings = avev3_get_settings,
  .set_settings = avev3_set_settings,
  .nway_reset = avev3_nway_reset,
  .get_link = avev3_get_link,
  .get_msglevel = avev3_get_msglevel,
  .set_msglevel = avev3_set_msglevel,
  .get_rx_csum = avev3_get_rx_csum,
  .get_tx_csum = avev3_get_tx_csum,
};


int avev3_open(struct net_device *dev)
{
  struct avev3_private *priv = netdev_priv(dev);
  unsigned long flags;

  if ( netif_msg_ifup(priv) ) {
    DBG_PRINT("ave_open()\n");
  }
  avev3_proc_add("ave_open");

  printk(KERN_INFO "avev3: %s %s\n", __DATE__, __TIME__);


  if( avev3_init_ring(dev) ){
    printk(KERN_ERR "avev3: <FAIL> avev3_init_ring() Failed");
    free_irq(dev->irq, dev);
    return -EAGAIN;
  }

  spin_lock_irqsave(&priv->lock, flags);
  avev3_global_init(dev);


  spin_unlock_irqrestore(&(priv->lock), flags);

  avev3_peaks_set_irq_lvl(dev->irq);

  mn_set_lateack_irq_type(dev->irq);

  if(request_irq(dev->irq, avev3_interrupt, SA_SHIRQ,
         dev->name, dev) !=0){
    printk(KERN_ERR "avev3: <FAIL> request_irq() Failed");
    return -EAGAIN;
  }


  netif_start_queue(dev);

  spin_lock_irqsave(&priv->lock, flags);
  mii_check_link(&priv->mii_if);
  spin_unlock_irqrestore(&(priv->lock), flags);

  if (netif_msg_ifup(priv)) {
    DBG_PRINT("%s: Done avev3_open().\n", dev->name);
  }

  priv->timer.expires = RUN_AT(_AVEV3_MNTR_TIME);
  priv->timer.data = (unsigned long)dev;
  priv->timer.function = &avev3_timer;  
  add_timer(&priv->timer);

#ifdef MODULE
  MOD_INC_USE_COUNT;
#endif 

  return 0;
}

int avev3_stop(struct net_device *dev)
{
  struct avev3_private *priv = netdev_priv(dev);
  unsigned long reg;
  unsigned long flags;

  if ( netif_msg_ifdown(priv) ) {
    DBG_PRINT("avev3_stop()\n");
  }
  avev3_proc_add("ave_stop");


  AVEV3_REG_READ(reg, _AVEV3_GIMR);
  AVEV3_REG_WRITE((reg & ~(_AVEV3_GI_RX)), _AVEV3_GIMR);

  del_timer_sync(&priv->timer);

  netif_stop_queue(dev);

  if (netif_msg_ifdown(priv)) {
    DBG_PRINT("%s: Shutting down AVEV3.\n", dev->name);
  }

  avev3_peaks_disable_irq(dev->irq);

#ifdef AVEV3_ENABLE_LONGWAIT_BY_SLEEP 

  spin_lock_irqsave(&priv->lock, flags);
  reg = (unsigned long)avev3_mdio_read(dev, priv->phy_id, MII_BMCR);

  avev3_flg_reset = 1;  
  spin_unlock_irqrestore(&priv->lock, flags);

  avev3_global_reset(dev);   

  spin_lock_irqsave(&priv->lock, flags);
  avev3_flg_reset = 0;  

  avev3_mdio_write(dev, priv->phy_id, MII_BMCR, reg);
  spin_unlock_irqrestore(&priv->lock, flags);

#else 

  spin_lock_irqsave(&priv->lock, flags);
  reg = (unsigned long)avev3_mdio_read(dev, priv->phy_id, MII_BMCR);

  avev3_global_reset(dev);

  avev3_mdio_write(dev, priv->phy_id, MII_BMCR, reg);
  spin_unlock_irqrestore(&priv->lock, flags);

#endif 

  avev3_term_ring(dev);

  AVEV3_DISABLE_INT();

  free_irq(dev->irq, dev);

#ifdef MODULE
  MOD_DEC_USE_COUNT;
#endif 



  return 0;
}


static int avev3_change_mtu(struct net_device *dev, int new_mtu)
{
  struct avev3_private *priv = NULL;
#ifdef  AVEV3_RX_MTU_CHANGE
  unsigned long reg;
#endif

  avev3_proc_add("avev3_change_mtu");
  if ( dev != NULL ) {
    priv = netdev_priv(dev);
    if ( netif_msg_drv(priv) ) {
      DBG_PRINT("avev3_change_mtu()\n");
    }
  } else {
    return -1;
  }


  if ( new_mtu + 18 > _AVEV3_MAX_ETHFRAME ) {
    if ( netif_msg_hw(priv) ) {
      DBG_PRINT("New MTU is too big.\n");
    }
    return -1;

  } else {
#ifdef  AVEV3_RX_MTU_CHANGE
    AVEV3_REG_READ(reg, _AVEV3_RXCR);
    reg &= 0xfffff800;
    reg |= (new_mtu + 18 & 0x7ff);
    AVEV3_REG_WRITE(reg, _AVEV3_RXCR);
#endif

    dev->mtu = new_mtu;
  }

  return 0;
}



int avev3_set_mac_address(struct net_device *dev, void *p)
{
  struct avev3_private *priv = netdev_priv(dev);
  struct sockaddr *addr = p;

  memcpy(dev->dev_addr, addr->sa_data, ETH_ALEN);

  priv->macaddr_got = 1;

  return(0);
}



int avev3_init_module(void)
{
  struct avev3_private *priv;
  int retval;
  struct avev3_descriptor_matrix desc_matrx = _AVEV3_DMATRX_INIT;

  DBG_PRINT("avev3_init_module()\n");
  avev3_proc_add("avev3_init_module");

  avev3_reg_base = (unsigned long)ioremap(_AVEV3_BASE, _AVEV3_IO_SIZE);
  if ((void *)avev3_reg_base == NULL) {
    printk(KERN_ERR "avev3: Could not ioremap device I/O.\n");
    return -1;
  }


  initialize_target();

  avev3_dev = alloc_etherdev(sizeof(struct avev3_private));
  if (avev3_dev == NULL) {
    printk(KERN_ERR "avev3: Could not allocate ethernet device.\n");
    return -1;
  }     

  priv = netdev_priv(avev3_dev);
  memset(priv, 0, sizeof(struct avev3_private));

#ifdef  _AVEV3_NETIF_MSG
  priv->msg_enable = DEBUG_DEFAULT;
#endif

  avev3_dev->base_addr = avev3_base;
  avev3_dev->irq = avev3_irq;

  avev3_global_reset(avev3_dev);

  avev3_get_version(avev3_dev);
  if ( strncmp(priv->id_str, AVEV3_ID, 4) != 0 ) {
    printk(KERN_ERR "avev3: unmatch ID charactors\n");
    goto err_free;
  }

  printk(KERN_INFO "  %s", drv_version);

  avev3_emac_init(avev3_dev);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,32) 
  avev3_dev->open           = avev3_open;
  avev3_dev->stop           = avev3_stop;
  avev3_dev->hard_start_xmit        = avev3_start_xmit;
  avev3_dev->do_ioctl           = avev3_ioctl;
  avev3_dev->set_multicast_list     = avev3_multicast;
  avev3_dev->get_stats          = avev3_stats;
  avev3_dev->change_mtu         = avev3_change_mtu;
#else 
  avev3_dev->netdev_ops = &avev3_netdev_ops;
  avev3_netdev_ops.ndo_open                 = avev3_open;
  avev3_netdev_ops.ndo_stop                 = avev3_stop;
  avev3_netdev_ops.ndo_start_xmit           = avev3_start_xmit;
  avev3_netdev_ops.ndo_do_ioctl             = avev3_ioctl;
  avev3_netdev_ops.ndo_set_multicast_list   = avev3_multicast;
  avev3_netdev_ops.ndo_get_stats            = avev3_stats;
  avev3_netdev_ops.ndo_change_mtu           = avev3_change_mtu;
  avev3_netdev_ops.ndo_set_mac_address      = avev3_set_mac_address;
#endif 

  SET_ETHTOOL_OPS(avev3_dev, &ethtool_ops);

  create_proc_read_entry(AVEV3_PROC_FILE, 0, NULL, avev3_proc_read, NULL);

#ifdef MODULE
  SET_MODULE_OWNER(avev3_dev);
#endif 

  priv->tx_dnum = desc_matrx.tx;
  priv->rx_dnum[0] = desc_matrx.rx[0];
  priv->mii_if.phy_id_mask  = 0x1f; 
  priv->mii_if.reg_num_mask = 0x1f; 
  priv->mii_if.phy_id       = AVEV3_PHY_ID & priv->mii_if.phy_id_mask;
  priv->mii_if.dev      = avev3_dev;    
  priv->mii_if.mdio_read    = avev3_mdio_read;
  priv->mii_if.mdio_write   = (void *)avev3_mdio_write;

  priv->phy_id = priv->mii_if.phy_id;   


  spin_lock_init(&(priv->lock));

  init_timer(&priv->timer); 

#ifdef RXFIFO_RESET_DISABLE
    printk("----------------------------------- \n");
    printk("----------------------------------- \n");
    printk("avev3: RxFIFO Reset disable when RxFIFO Overflow\n");
    printk("----------------------------------- \n");
    printk("----------------------------------- \n");
#endif 

#ifdef DPRAM_ACCESS_TWICE
    printk("----------------------------------- \n");
    printk("----------------------------------- \n");
    printk("avev3: DPRAM Access twice when receive packet.\n");
    printk("----------------------------------- \n");
    printk("----------------------------------- \n");
#endif 


  retval = register_netdev(avev3_dev);
  if ( retval != 0 ) {
    printk(KERN_ERR "avev3: <FAIL> register netdevice failed.\n");
    goto err_free;
  }

  return 0;

 err_free:
  free_netdev(avev3_dev);   
  avev3_dev = NULL;
  return -1;
}

void avev3_cleanup_module(void)
{
  struct avev3_private *priv = netdev_priv(avev3_dev);

  if ( netif_msg_drv(priv) ) {
    DBG_PRINT("avev3_cleanup_module()\n");
  }
  avev3_proc_add("avev3_cleanup_module");

  unregister_netdev(avev3_dev);
  free_netdev(avev3_dev);   
  avev3_dev = NULL;


  remove_proc_entry(AVEV3_PROC_FILE, NULL);

  iounmap((void *)avev3_reg_base);
}


#if 0 
static int avev3_pm_probe(struct platform_device *pdev)
{
  struct avev3_private *priv;
  struct resource *res, *irq_res;
  int res_size;
  int ret;

  if (!pdev->dev.platform_data) {
    printk(KERN_ERR"%s: platform_data not provided\n", AVEV3_CHDEV_NAME);
    return(-ENODEV);
  }

  res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "avev3-memory");
  if(!res){
    res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
  }
  if(!res){
    printk("%s: Could not allocate resource.\n", AVEV3_CHDEV_NAME);
    return(-ENODEV);
  }
  res_size = resource_size(res);

  irq_res = platform_get_resource(pdev, IORESOURCE_IRQ, 0);
  if(!irq_res){
    printk("%s: Could not allocate irq resource.\n", AVEV3_CHDEV_NAME);
    return(-ENODEV);
  }

  if(!request_mem_region(res->start, res_size, AVEV3_CHDEV_NAME)) {
    return(-EBUSY);
  }

  ret = avev3_init_module();
  if(ret != 0){
    platform_set_drvdata(pdev, NULL);
    release_mem_region(res->start, resource_size(res));
  }


  SET_NETDEV_DEV(avev3_dev, &pdev->dev);

  avev3_dev->irq = irq_res->start; 

  priv = netdev_priv(avev3_dev);

  platform_set_drvdata(pdev, avev3_dev);

  return(0);
}
static int __devexit avev3_pm_remove(struct platform_device *pdev)
{
  struct resource *res;

  res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "avev3-memory");
  if(!res){
    res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
  }
  release_mem_region(res->start, resource_size(res));

  avev3_cleanup_module();
  return(0);
}
static int avev3_pm_suspend(struct device *dev)
{
  return(0);
}
static int avev3_pm_resume(struct device *dev)
{
  return(0);
}
static struct dev_pm_ops avev3_pm_ops = {
        .suspend        = avev3_pm_suspend,
        .resume         = avev3_pm_resume,
};
#define AVEV3_PM_OPS (&avev3_pm_ops)
static struct platform_driver avev3_pm_driver = {
        .probe = avev3_pm_probe,
        .remove = __devexit_p(avev3_pm_remove),
        .driver = {
                .name   = AVEV3_CHDEV_NAME,
                .owner  = THIS_MODULE,
                .pm     = AVEV3_PM_OPS,
        },
};
int avev3_pm_init_module(void)
{
  return( platform_driver_register(&avev3_pm_driver) );
}
void avev3_pm_cleanup_module(void)
{
  platform_driver_unregister(&avev3_pm_driver);
  return;
}
module_init(avev3_pm_init_module);
module_exit(avev3_pm_cleanup_module);
#else 
module_init(avev3_init_module);
module_exit(avev3_cleanup_module);
#endif 
